// =============================
// 一键绑定登录态（Shopee / Lazada）popup.js
// 关键点：按 URL 采集、跨域合并、去重时保留“更晚过期”的 Cookie
// =============================

// ---------- 小工具 ----------
function $(id) { return document.getElementById(id); }

function fmtDate(dt) {
  const y = dt.getFullYear();
  const m = String(dt.getMonth() + 1).padStart(2, '0');
  const d = String(dt.getDate()).padStart(2, '0');
  const hh = String(dt.getHours()).padStart(2, '0');
  const mm = String(dt.getMinutes()).padStart(2, '0');
  return `${y}-${m}-${d} ${hh}:${mm}`;
}

// ---------- TTL 估算 ----------
function daysLeftFromCookies(cookies) {
  const now = Date.now() / 1000; // 秒
  // 取有过期时间的 cookie
  const exps = cookies
    .map(c => c.expirationDate ?? c.expires)
    .filter(ts => typeof ts === 'number' && ts > now);

  if (exps.length === 0) return null;

  // 忽略过短 TTL（例如 < 12 小时）的 cookie（很多是分析/防刷用的）
  const filtered = exps.filter(ts => (ts - now) >= 12 * 3600);
  const pool = filtered.length > 0 ? filtered : exps;

  const minExp = Math.min(...pool);
  const days = Math.max(0, Math.ceil((minExp - now) / 86400)); // 向上取整
  return { days, expiresAt: new Date(minExp * 1000) };
}

function showTTL(cookies) {
  const box = $('ttl');
  const est = daysLeftFromCookies(cookies);
  if (!box) return;
  if (!est) {
    box.style.display = 'block';
    box.className = 'help err';
    box.textContent = '未找到带过期时间的 Cookie，可能会话型 Cookie 很短或需先登录站点。';
    return;
  }
  const { days, expiresAt } = est;
  box.style.display = 'block';
  box.className = 'help ok';
  box.innerHTML = `本地估算：<span class="pill">剩余 ${days} 天</span> 最早过期：<span class="pill">${fmtDate(expiresAt)}</span>`;
  return est;
}

// ---------- 采集 URL 列表（按 URL 采更稳） ----------
const URL_MAP = {
  shopee: [
    'https://shopee.com.my',
    'https://www.shopee.com.my',
    'https://accounts.shopee.com',
    'https://shopee.com'
  ],
  lazada: [
    'https://www.lazada.com.my',
    'https://member.lazada.com.my',
    'https://my.lazada.com.my'
  ]
};

// 用 URL 取 cookie（比按 domain 更能覆盖关键登录态）
function getCookiesByUrl(url) {
  return new Promise((resolve) => {
    chrome.cookies.getAll({ url }, (cks) => resolve(cks || []));
  });
}

// 统一成后端期望的字段
function normalizeCookie(ck) {
  return {
    name: ck.name,
    value: ck.value,
    domain: ck.domain,
    path: ck.path || '/',
    expirationDate: ck.expirationDate,   // 可能 undefined
    httpOnly: !!ck.httpOnly,
    secure: !!ck.secure,
    sameSite: (ck.sameSite || 'unspecified').toLowerCase(),
    // 兼容字段
    expires: ck.expirationDate
  };
}

// 采集并合并（同 name+domain+path 只保留过期更晚的）
async function collectCookies(platform) {
  const urls = URL_MAP[platform] || [];
  let all = [];
  for (const u of urls) {
    try {
      const arr = await getCookiesByUrl(u);
      if (arr && arr.length) all = all.concat(arr);
    } catch (_) {}
  }
  const map = new Map();
  for (const c of all) {
    const key = `${c.name}|${c.domain}|${c.path || '/'}`;
    const old = map.get(key);
    if (!old || (c.expirationDate ?? 0) >= (old?.expirationDate ?? 0)) {
      map.set(key, c);
    }
  }
  return Array.from(map.values()).map(normalizeCookie);
}

// ---------- 表单读写 ----------
function loadForm() {
  chrome.storage.local.get(['server', 'token', 'platform', 'account'], (cfg) => {
    if (cfg.server)   $('server').value   = cfg.server;
    if (cfg.token)    $('token').value    = cfg.token;
    if (cfg.platform) $('platform').value = cfg.platform;
    if (cfg.account)  $('account').value  = cfg.account;
  });
}

function saveForm() {
  const server   = $('server').value.trim();
  const token    = $('token').value.trim();
  const platform = $('platform').value.trim();
  const account  = $('account').value.trim();
  chrome.storage.local.set({ server, token, platform, account });
}

// ---------- 上传到服务端 ----------
async function upload(server, token, platform, account, cookies, clientTTL) {
  const url = `${server.replace(/\/+$/,'')}/api/cookies/import`;
  const payload = {
    token, platform, account, cookies,
    client_days_left: clientTTL?.days ?? null,
    client_expires_at: clientTTL?.expiresAt ? Math.floor(clientTTL.expiresAt.getTime()/1000) : null
  };
  const res = await fetch(url, {
    method: 'POST',
    headers: { 'Content-Type':'application/json' },
    body: JSON.stringify(payload)
  });
  if (!res.ok) {
    const txt = await res.text().catch(()=> '');
    throw new Error(`上传失败（HTTP ${res.status}）：${txt || '未知错误'}`);
  }
  return res.json();
}

// ---------- 事件绑定 ----------
document.addEventListener('DOMContentLoaded', () => {
  loadForm();

  // 输入变化自动保存（可选）
  ['server','token','platform','account'].forEach(id => {
    const el = $(id);
    if (el) el.addEventListener('change', saveForm);
  });

  const out = $('result');
  const btn = $('btnFetch');

  if (btn) btn.addEventListener('click', async () => {
    if (out) { out.textContent = ''; out.className = 'help'; }
    if ($('ttl')) $('ttl').style.display = 'none';

    const server   = $('server').value.trim();
    const token    = $('token').value.trim();
    const platform = $('platform').value.trim();
    const account  = $('account').value.trim();

    // 校验
    if (!server || !/^https?:\/\//i.test(server)) {
      out.className = 'help err';
      out.textContent = '请填写正确的服务端地址（http/https）。';
      return;
    }
    if (!token) {
      out.className = 'help err';
      out.textContent = '请填写 IMPORT_TOKEN。';
      return;
    }
    if (!platform) {
      out.className = 'help err';
      out.textContent = '请选择平台（Shopee/Lazada）。';
      return;
    }
    if (!account) {
      out.className = 'help err';
      out.textContent = '请填写账号标识（如 buyer1）。';
      return;
    }

    saveForm();

    // 按钮 loading
    const oldTxt = btn.textContent;
    btn.disabled = true;
    btn.textContent = '正在采集并上传…';

    try {
      out.className = 'help';
      out.textContent = '正在采集 Cookies…（请确保此浏览器已登录并打开订单页）';

      const cookies = await collectCookies(platform);
      if (!cookies.length) {
        out.className = 'help err';
        out.textContent = '没有采集到任何 Cookie。请先在此浏览器登录相应站点，然后重试。';
        return;
      }

      const ttl = showTTL(cookies); // 显示本地估算

      out.className = 'help';
      out.textContent = `采集到 ${cookies.length} 条 Cookie，正在上传到服务端…`;

      const resp = await upload(server, token, platform, account, cookies, ttl || null);

      if (resp && resp.ok) {
        // 服务端估算（若返回）
        const sDays = resp.days_left;
        const sExp  = resp.expires_at;
        const add   = (sDays != null || sExp)
          ? `（服务端估算：剩余 ${sDays ?? '-'} 天，最早过期 ${sExp ?? '-'}）`
          : '';
        out.className = 'help ok';
        out.textContent = `绑定成功！已保存登录状态。${add}`;
      } else {
        out.className = 'help err';
        out.textContent = `上传失败：${(resp && resp.msg) ? resp.msg : '未知错误'}`;
      }
    } catch (e) {
      out.className = 'help err';
      out.textContent = e?.message || String(e);
    } finally {
      btn.disabled = false;
      btn.textContent = oldTxt;
    }
  });
});
